/*
 * 2004 Jan Struyf
 *
 */

#include <string.h>
#include <string>
#include <vector>
#include <iostream>
using namespace std;

#include "tokens/stokenizer.h"
#include "cmdline.h"
#include "cutils.h"

CmdLineOptionArg::CmdLineOptionArg(const char* name) {
	m_Name = name;
	m_MinCard = -1;
	m_MaxCard = -1;
	m_Card = 0;
}

CmdLineOptionArg::~CmdLineOptionArg() {
}

void CmdLineOptionArg::showExtraHelp() {
}

void CmdLineOptionArg::initArg() {
}

void CmdLineOptionArg::setDefaultValue() {
}

void CmdLineOptionArg::initShowError() {
	cerr << ">> Option " << getObject()->getOptionPrefix() << getOption()->getName();
	if (getOption()->getMaxNbArgs() != 1) {
		cerr << " argument '" << getName() << "'";
	}
}

bool CmdLineOptionArg::appendValue(const string& value) {
	return addValue(value);
}

CmdLineOptionList* CmdLineOptionArg::getObject() {
	return m_Option->getObject();
}

CmdLineArgString::CmdLineArgString(const char* name) : CmdLineOptionArg(name) {
	setMaxCard(1);
}

CmdLineArgString::~CmdLineArgString() {
}

void CmdLineArgString::write(ostream& os) {
	os << "\"" << m_Value << "\"";
}

void CmdLineArgString::reset() {
	m_Value = "";
	m_Card = 0;
}

bool CmdLineArgString::addValue(const string& value) {
	m_Value = value;
	str_remove_quote(m_Value);
	m_Card++;
	return true;
}

bool CmdLineArgString::appendValue(const string& value) {
	if (m_Value == "") {
		m_Value = value;
		str_remove_quote(m_Value);
	} else {
		string temp = value;
		str_remove_quote(temp);
		m_Value += string(" ") + temp;
	}
	m_Card++;
	return true;
}

void CmdLineArgString::setDefaultValue() {
	m_Value = m_Default;
	m_Card++;
}

bool CmdLineArgString::isDefault() {
	return m_Value == m_Default;
}

CmdLineArgInt::CmdLineArgInt(const char* name) : CmdLineOptionArg(name) {
	setMaxCard(1);
}

CmdLineArgInt::~CmdLineArgInt() {
}

void CmdLineArgInt::write(ostream& os) {
	os << m_Value;
}

void CmdLineArgInt::reset() {
	m_Value = 0;
	m_Card = 0;
}

bool CmdLineArgInt::addValue(const string& value) {
	m_Value = atoi(value.c_str());
	m_Card++;
	return true;
}

void CmdLineArgInt::setDefaultValue() {
	m_Value = m_Default;
	m_Card++;
}

bool CmdLineArgInt::isDefault() {
	return m_Value == m_Default;
}

CmdLineArgSet::CmdLineArgSet(const char* name) : CmdLineOptionArg(name) {
}

CmdLineArgSet::~CmdLineArgSet() {
}

bool CmdLineArgSet::addValue(const string& value) {
	for (int i = 0; i < m_Values.size(); i++) {
		if (str_i_equals(m_Values[i], value)) {
			if (m_HasValue[i] == CMDLINE_NO) {
				m_HasValue[i] = CMDLINE_YES;
				m_Card++;
				return true;
			}
		}
	}
	initShowError();
	cerr << " illegal value '" << value << "'" << endl;
	return false;
}

void CmdLineArgSet::write(ostream& os) {
	bool prev = false;
	for (int i = 0; i < m_Values.size(); i++) {
		if (m_HasValue[i] == CMDLINE_YES) {
			if (prev) { os << " "; } else { prev = true; }
			os << m_Values[i];
		}
	}
}

void CmdLineArgSet::reset() {
	for (int i = 0; i < m_Values.size(); i++) {
		if (m_HasValue[i] != CMDLINE_UNSUPPORTED) {
			m_HasValue[i] = CMDLINE_NO;
		}
	}
	m_Card = 0;
}

void CmdLineArgSet::setDefaultValue() {
	for (int i = 0; i < m_Defaults.size(); i++) {
		m_HasValue[m_Defaults[i]] = CMDLINE_YES;
		m_Card++;
	}
}

bool CmdLineArgSet::isDefault() {
	for (int i = 0; i < m_Values.size(); i++) {
		if (m_HasValue[i] != CMDLINE_UNSUPPORTED) {
			bool yes = m_HasValue[i] == CMDLINE_YES;
			bool default_yes = false;
			for (int j = 0; j < m_Defaults.size(); j++) {
				if (m_Defaults[j] == i) {
					default_yes = true;
				}
			}
			if (yes != default_yes) return false;
		}
	}
	return true;
}

void CmdLineArgSet::showExtraHelp() {
	cerr << "   Possible values: ";
	for (int i = 0; i < m_Values.size(); i++) {
		if (m_HasValue[i] != CMDLINE_UNSUPPORTED) {
			if (i != 0) cerr << ',';
			cerr << m_Values[i];
		}
	}
	cerr << endl;
}

void CmdLineArgSet::addPossibleValue(const char* value) {
	m_Values.push_back(string(value));
	m_HasValue.push_back(CMDLINE_NO);
}

CmdLineOption::CmdLineOption(const char* name) {
	addAlias(name);
	initialize();
}

CmdLineOption::CmdLineOption(const char* name, const char* alias) {
	addAlias(name); addAlias(alias);
	initialize();
}

CmdLineOption::CmdLineOption(const char* name, const char* alias1, const char* alias2) {
	addAlias(name); addAlias(alias1); addAlias(alias2);
	initialize();
}

CmdLineOption::~CmdLineOption() {
	deleteArgs();
}

void CmdLineOption::initialize() {
	m_HasOption = false;
	m_Expert = false;
	m_MinNbArgs = 0;
	m_NbArgs = 0;
}

void CmdLineOption::deleteArgs() {
	for (int i = 0; i < m_Args.size(); i++) {
		if (m_Args[i] != NULL) {
			delete m_Args[i];
			m_Args[i] = NULL;
		}
	}
}

void CmdLineOption::initOption() {
	for (int i = 0; i < m_Args.size(); i++) {
		if (m_Args[i] != NULL) {
			m_Args[i]->initArg();
		}
	}
}

void CmdLineOption::setDefaultValues() {
	for (int i = 0; i < m_Args.size(); i++) {
		if (m_Args[i] != NULL) {
			m_Args[i]->setDefaultValue();
		}
	}
}

bool CmdLineOption::allDefaults() {
	for (int i = 0; i < m_Args.size(); i++) {
		if (m_Args[i] != NULL) {
			if (!m_Args[i]->isDefault()) return false;
		}
	}
	return true;
}

void CmdLineOption::addArg(CmdLineOptionArg* arg) {
	m_Args.push_back(arg);
	arg->setOption(this);
}

void CmdLineOption::addAlias(const char* alias) {
	m_Names.push_back(string(alias));
}

void CmdLineOption::showHelp() {
	cerr << "Option: " << getObject()->getOptionPrefix() << getName();
	for (int i = 0; i < getMaxNbArgs(); i++) {
		if (i >= getMinNbArgs()) {
			cerr << " [" << getArg(i)->getName() << "]";
		} else {
			cerr << " " << getArg(i)->getName();
		}
	}
	cerr << endl;
	for (int i = 0; i < getMaxNbArgs(); i++) {
		CmdLineOptionArg* argi = getArg(i);
		cerr << "   Argument '" << argi->getName() << "': " << argi->getHelp() << endl;
		argi->showExtraHelp();
	}
}

CmdLineOptionList::CmdLineOptionList() {
	m_Error = 0;
}

CmdLineOptionList::~CmdLineOptionList() {
	deleteOptions();
}

void CmdLineOptionList::deleteOptions() {
	for (int i = 0; i < m_Options.size(); i++) {
		if (m_Options[i] != NULL) {
			delete m_Options[i];
			m_Options[i] = NULL;
		}
	}
}

void CmdLineOptionList::initOptions() {
	for (int i = 0; i < m_Options.size(); i++) {
		if (m_Options[i] != NULL) {
			m_Options[i]->initOption();
		}
	}
}

void CmdLineOptionList::setDefaultValues() {
	for (int i = 0; i < m_Options.size(); i++) {
		CmdLineOption* opt = m_Options[i];
		if (opt != NULL && !opt->hasOption()) {
			opt->setDefaultValues();
		}
	}
}

bool CmdLineOptionList::allDefaults() {
	for (int i = 0; i < m_Options.size(); i++) {
		if (m_Options[i] != NULL) {
			if (!m_Options[i]->allDefaults()) return false;
		}
	}
	return true;
}

CmdLineOption* CmdLineOptionList::getOption(const string& name) {
	for (int i = 0; i < m_Options.size(); i++) {
		CmdLineOption* opt = m_Options[i];
		if (opt != NULL) {
			for (int j = 0; j < opt->getNbNames(); j++) {
				if (str_i_equals(opt->getName(j), name)) {
					return opt;
				}
			}
		}
	}
	return NULL;
}

void CmdLineOptionList::showHelp(int helpid) {
	bool expert = false;
	CmdLineOption* help = getOption(helpid);
	CmdLineArgString* item = (CmdLineArgString*)help->getArg(0);
	if (item->getCard() == 1) {
		const string& value = item->getValue();
		if (value == "expert") {
			expert = true;
		} else {
			CmdLineOption* opt = getOption(value);
			if (opt == NULL) {
				cerr << ">> Unknown option '" << getOptionPrefix() << value << "'" << endl;
			} else {
				cerr << endl;
				opt->showHelp();
			}
			return;
		}
	}
	cerr << endl << "Options:" << endl;
	for (int i = 0; i < m_Options.size(); i++) {
		CmdLineOption* opt = m_Options[i];
		if (opt != NULL && (!opt->isExpert() || expert)) {
			string str = " ";
			str += getOptionPrefix();
			str += opt->getName();
			cerr << str;
			for (int i = str.length(); i < 15; i++) cerr << ' ';
			cerr << opt->getHelp() << endl;
		}
	}
	if (!expert) cerr << endl << "To show expert options: gle " << getOptionPrefix() << "help expert" << endl;
}

bool CmdLineOptionList::hasOption(int id) {
	if (id >= m_Options.size()) return false;
	if (m_Options[id] == NULL) return false;
	return m_Options[id]->hasOption();
}

char CmdLineOptionList::getOptionPrefix() {
#ifdef __WIN32__
	return '/';
#else
	return '-';
#endif
}

void CmdLineOptionList::addOption(CmdLineOption* option, int id) {
	int size = m_Options.size();
	if (id >= size) {
		m_Options.reserve(id+1);
		for (int i = size; i <= id; i++) m_Options.push_back(NULL);
	}
	option->setObject(this);
	m_Options[id] = option;
}

CmdLineArgString* CmdLineOptionList::addStringOption(const char* name, int id) {
	CmdLineOption* opt = new CmdLineOption(name);
	CmdLineArgString* arg = new CmdLineArgString("");
	opt->addArg(arg);
	addOption(opt, id);
	return arg;
}

ConfigSection::ConfigSection(const char* name) {
	m_Name = name;
}

ConfigSection::~ConfigSection() {
}

ConfigCollection::ConfigCollection() {
}

ConfigCollection::~ConfigCollection() {
	deleteSections();
}

void ConfigCollection::deleteSections() {
	for (int i = 0; i < m_Sections.size(); i++) {
		if (m_Sections[i] != NULL) {
			delete m_Sections[i];
			m_Sections[i] = NULL;
		}
	}
}

void ConfigCollection::setDefaultValues() {
	for (int i = 0; i < m_Sections.size(); i++) {
		if (m_Sections[i] != NULL) {
			m_Sections[i]->setDefaultValues();
		}
	}
}

bool ConfigCollection::allDefaults() {
	for (int i = 0; i < m_Sections.size(); i++) {
		if (m_Sections[i] != NULL) {
			if (!m_Sections[i]->allDefaults()) return false;
		}
	}
	return true;
}

void ConfigCollection::addSection(ConfigSection* section, int id) {
	int size = m_Sections.size();
	if (id >= size) {
		m_Sections.reserve(id+1);
		for (int i = size; i <= id; i++) m_Sections.push_back(NULL);
	}
	m_Sections[id] = section;
}

ConfigSection* ConfigCollection::getSection(const string& name) {
	for (int i = 0; i < m_Sections.size(); i++) {
		ConfigSection* sec = m_Sections[i];
		if (sec != NULL && str_i_equals(sec->getName(), name)) {
			return sec;
		}
	}
	return NULL;
}

CmdLineObj::CmdLineObj() {
	m_ArgC = 0;
	m_ArgIdx = 0;
	m_ArgV = NULL;
}

CmdLineObj::~CmdLineObj() {
}

const char* CmdLineObj::getNextArg() {
	return m_ArgIdx >= m_ArgC ? NULL : m_ArgV[m_ArgIdx++];
}

void CmdLineObj::parse(int argc, char** argv) {
	const char* crarg;
	m_ArgC = argc;
	m_ArgV = argv;
	m_ArgIdx = 1;
	int argidx = 0;
	bool inmainargs = false;
	CmdLineOption* cropt = NULL;
	while ((crarg = getNextArg()) != NULL) {
		int len = strlen(crarg);
#ifdef __WIN32__
		/* Also allow options with prefix '/' on Windows */
		/* On Unix, '/' might be the start of an absolute path !! */
		if (len >= 2 && (crarg[0] == '-' || crarg[0] == '/')) {
#else
		if (len >= 2 && crarg[0] == '-') {
#endif
			string optionname;
			if (crarg[1] == '-') {
				optionname = crarg+2;
			} else {
				optionname = crarg+1;
			}
			if (inmainargs) {
				cerr << ">> Options should come before " << m_MainArgType << " arguments" << endl;
				m_Error = 1;
				return;
			}
			if (cropt != NULL) {
				if (argidx < cropt->getMinNbArgs()) {
					cerr << ">> Option '" << cropt->getName() << "' requires "
					     << cropt->getNbArgs() << " arguments" << endl;
					m_Error = 1;
					return;
				}
			}
			cropt = getOption(optionname);
			if (cropt == NULL) {
				cerr << ">> Unknown option '" << getOptionPrefix() << optionname << "'" << endl;
				m_Error = 1;
				return;
			} else {
				argidx = 0;
				cropt->setHasOption(true);
			}
		} else {
			if (cropt != NULL && argidx < cropt->getMaxNbArgs()) {
				/* Assume extra argument for current option */
				addOptionArg(cropt, argidx, string(crarg));
				if (hasError()) return;
				argidx++;
			} else {
				/* Assume main argument */
				inmainargs = true;
				m_MainArgs.push_back(string(crarg));
			}
		}
	}
	setDefaultValues();
}

void CmdLineObj::addOptionArg(CmdLineOption* cropt, int argidx, const string& crarg) {
	CmdLineOptionArg* arg = cropt->getArg(argidx);
	char_separator separator(",", "", drop_empty_tokens);
	tokenizer<char_separator> tokens(crarg, separator);
	while (tokens.has_more()) {
		if (arg->getMaxCard() == -1 || arg->getCard() < arg->getMaxCard()) {
			if (!arg->addValue(tokens.next_token())) {
				m_Error = 1;
			}
		} else {
			cerr << ">> Option '" << getOptionPrefix() << cropt->getName() << "'";
			if (cropt->getMaxNbArgs() > 1) {
				cerr << " argument " << argidx << " (" << arg->getName() << ")";
			}
			cerr << " takes at most " << arg->getMaxCard() << " value(s)" << endl;
			m_Error = 1;
		}
	}
}

