/*
 * 2004 Jan Struyf
 *
 */

#ifndef INCLUDE_CMDLINE
#define INCLUDE_CMDLINE

class CmdLineOptionList;
class CmdLineOption;

class CmdLineOptionArg {
protected:
	int m_MinCard, m_MaxCard, m_Card;
	string m_Help, m_Name;
	CmdLineOption* m_Option;
public:
	CmdLineOptionArg(const char* name);
	virtual ~CmdLineOptionArg();
	virtual bool isDefault() = 0;
	virtual void reset() = 0;
	virtual void write(ostream& os) = 0;
	virtual bool addValue(const string& value) = 0;
	virtual bool appendValue(const string& value);
	virtual void showExtraHelp();
	virtual void initArg();
	virtual void setDefaultValue();
	void initShowError();
	CmdLineOptionList* getObject();
	inline int getCard() { return m_Card; }
	inline int getMinCard() { return m_MinCard; }
	inline int getMaxCard() { return m_MaxCard; }
	inline void setCardLimits(int min, int max) { m_MinCard = min; m_MaxCard = max; }
	inline void setMinCard(int card) { m_MinCard = card; }
	inline void setMaxCard(int card) { m_MaxCard = card; }
	inline const string& getHelp() { return m_Help; }
	inline void setHelp(const char* help) { m_Help = help; }
	inline const string& getName() { return m_Name; }
	inline CmdLineOption* getOption() { return m_Option; }
	inline void setOption(CmdLineOption* option) { m_Option = option; }
};

class CmdLineArgString : public CmdLineOptionArg {
protected:
	string m_Value;
	string m_Default;
public:
	CmdLineArgString(const char* name);
	~CmdLineArgString();
	virtual bool isDefault();
	virtual void reset();
	virtual void write(ostream& os);
	virtual bool addValue(const string& value);
	virtual bool appendValue(const string& value);
	virtual void setDefaultValue();
	inline const string& getValue() { return m_Value; }
	inline void setDefault(const char* value) { m_Default = value; }
	inline const string& getDefault() { return m_Default; }
	inline string* getValuePtr() { return &m_Value; }
};

class CmdLineArgInt : public CmdLineOptionArg {
protected:
	int m_Value;
	int m_Default;
public:
	CmdLineArgInt(const char* name);
	~CmdLineArgInt();
	virtual bool isDefault();
	virtual void reset();
	virtual void write(ostream& os);
	virtual bool addValue(const string& value);
	virtual void setDefaultValue();
	inline const int getValue() { return m_Value; }
	inline void setDefault(int value) { m_Default = value; }
};

#define CMDLINE_NO          0
#define CMDLINE_YES         1
#define CMDLINE_UNSUPPORTED 2

class CmdLineArgSet : public CmdLineOptionArg {
protected:
	vector<string> m_Values;
	vector<int> m_HasValue;
	vector<int> m_Defaults;
public:
	CmdLineArgSet(const char* name);
	~CmdLineArgSet();
	virtual bool isDefault();
	virtual void reset();
	virtual void write(ostream& os);
	virtual bool addValue(const string& value);
	virtual void showExtraHelp();
	virtual void setDefaultValue();
	void addPossibleValue(const char* value);
	inline void setUnsupportedValue(int id) { m_HasValue[id] = CMDLINE_UNSUPPORTED; }
	inline void addDefaultValue(int id) { m_Defaults.push_back(id); }
	inline int getNbValues() { return m_Values.size(); }
	inline bool hasValue(int id) { return m_HasValue[id] == CMDLINE_YES; }
};

class CmdLineOption {
protected:
	bool m_HasOption, m_Expert;
	int m_MinNbArgs, m_NbArgs;
	vector<string> m_Names;
	vector<CmdLineOptionArg*> m_Args;
	CmdLineOptionList* m_Object;
	string m_Help;
public:
	CmdLineOption(const char* name);
	CmdLineOption(const char* name, const char* alias);
	CmdLineOption(const char* name, const char* alias1, const char* alias2);
	~CmdLineOption();
	void initialize();
	void deleteArgs();
	void addAlias(const char* alias);
	void showHelp();
	void initOption();
	void setDefaultValues();
	void addArg(CmdLineOptionArg* arg);
	bool allDefaults();
	inline int getNbNames() { return m_Names.size(); }
	inline const string& getName(int i) { return m_Names[i]; }
	inline const string& getName() { return m_Names[0]; }
	inline int getMinNbArgs() { return m_MinNbArgs; }
	inline int getMaxNbArgs() { return m_Args.size(); }
	inline int getNbArgs() { return m_NbArgs; }
	inline CmdLineOptionArg* getArg(int i) { return m_Args[i]; }
	inline bool hasOption() { return m_HasOption; }
	inline void setHasOption(bool has) { m_HasOption = has; }
	inline void setMinNbArgs(int min) { m_MinNbArgs = min; }
	inline const string& getHelp() { return m_Help; }
	inline void setHelp(const char* help) { m_Help = help; }
	inline bool isExpert() { return m_Expert; }
	inline void setExpert(bool expert) { m_Expert = expert; }
	inline CmdLineOptionList* getObject() { return m_Object; }
	inline void setObject(CmdLineOptionList* obj) { m_Object = obj; }
};

class CmdLineOptionList {
protected:
	vector<CmdLineOption*> m_Options;
	int m_Error;
public:
	CmdLineOptionList();
	~CmdLineOptionList();
	void addOption(CmdLineOption* option, int id);
	CmdLineOption* getOption(const string& name);
	CmdLineArgString* addStringOption(const char* name, int id);
	void deleteOptions();
	char getOptionPrefix();
	void showHelp(int helpid);
	void initOptions();
	void setDefaultValues();
	bool allDefaults();
	bool hasOption(int id);
	inline int hasError() { return m_Error == 1; }
	inline int getNbOptions() { return m_Options.size(); }
	inline CmdLineOption* getOption(int i) { return m_Options[i]; }
	inline CmdLineOptionArg* getOptionValue(int i) { return m_Options[i]->getArg(0); }
};

class ConfigSection : public CmdLineOptionList {
protected:
	string m_Name;
public:
	ConfigSection(const char* name);
	~ConfigSection();
	inline const string& getName() { return m_Name; }
	inline void setName(const char* name) { m_Name = name; }
};

class ConfigCollection {
protected:
	vector<ConfigSection*> m_Sections;
public:
	ConfigCollection();
	~ConfigCollection();
	void deleteSections();
	bool allDefaults();
	void setDefaultValues();
	void addSection(ConfigSection* section, int id);
	ConfigSection* getSection(const string& name);
	inline int getNbSections() { return m_Sections.size(); }
	inline ConfigSection* getSection(int id) { return m_Sections[id]; };
};

class CmdLineObj : public CmdLineOptionList {
protected:
	string m_MainArgType;
	vector<string> m_MainArgs;
	int m_ArgC, m_ArgIdx;
	char** m_ArgV;
public:
	CmdLineObj();
	~CmdLineObj();
	void parse(int argc, char** argv);
	const char* getNextArg();
	void addOptionArg(CmdLineOption* cropt, int argidx, const string& crarg);
	inline int getNbMainArgs() { return m_MainArgs.size(); }
	inline const string& getMainArg(int i) { return m_MainArgs[i]; }
	inline vector<string>* getMainArgs() { return &m_MainArgs; }
	inline void setMainArgType(const char* type) { m_MainArgType = type; }
};

#endif
