/*
 * 2004 Jan Struyf
 *
 */

#include <math.h>
#include <stdlib.h>
#include <time.h>
#include <string>
#include <sstream>
#include <iostream>
#include <vector>
#include <list>
#include <map>
#include <fstream>
#ifdef __WIN32__
	#include <windows.h>
#endif

using namespace std;

#include "SourceLine.h"
#include "all.h"
#include "tokens/Tokenizer.h"
#include "mem_limits.h"
#include "token.h"
#include "core.h"
#include "mygraph.h"
#include "file_io.h"
#include "texinterface.h"
#include "cutils.h"
#include "cmdline.h"
#include "gprint.h"
#include "config.h"
#include "drawit.h"

#define BEGINDEF extern
#include "begin.h"

extern string GLE_TOP_DIR;
extern _GLESource GLESource;
extern ConfigCollection g_Config;

void doskip(char *s,int *ct);
char *un_quote(char *ct);

#define skipspace doskip(tk[ct],&ct)

void begin_config(const char* block, int *pln, int *pcode, int *cp) {
	string block_name(block);
	ConfigSection* section = g_Config.getSection(block_name);
	if (section == NULL) {
		gprint("Unrecognised CONFIG section {%s}\n ", block_name.c_str());
	}
	// Start with pcode from the next line
	(*pln)++;
	begin_init();
	while (true) {
		int st = begin_token(&pcode,cp,pln,srclin,tk,&ntk,outbuff);
		if (!st) {
			/* exit loop */
			break;
		}
		int ct = 1;
		int mode = 0;
		bool plus_is = false;
		CmdLineOption* option = NULL;
		while (ct <= ntk) {
			skipspace;
			if (section != NULL) {
				if (mode == 0) {
					option = section->getOption(tk[ct]);
					if (option == NULL) {
						gprint("Not a valid setting for section '%s': {%s}\n", block_name.c_str(), tk[ct]);
					}
				} else if (mode == 1) {
					if (strcmp(tk[ct], "=") == 0) {
						plus_is = false;
					} else if (strcmp(tk[ct], "+=") == 0) {
						plus_is = true;
					} else {
						gprint("Expected '=' or '+=', not {%s}\n", tk[ct]);
					}
				} else if (option != NULL) {
					CmdLineOptionArg* arg = option->getArg(0);
					if (!plus_is) arg->reset();
					arg->appendValue(tk[ct]);
				}
				mode++;
			}
			ct++;
		}
	}
}

void init_config(ConfigCollection* collection) {
	ConfigSection* section;
	CmdLineOption* option;
	CmdLineArgString* strarg;
	CmdLineArgSet* setarg;
	section = new ConfigSection("tools");
	/* LaTeX */
	strarg = section->addStringOption("tex", GLE_TOOL_LATEX_CMD);
#ifdef __WIN32__
	strarg->setDefault("latex.exe");
#endif
#ifdef __UNIX__
	strarg->setDefault("latex");
#endif
#ifdef __OS2__
	strarg->setDefault("vlatexp.cmd");
#endif
	/* PdfLaTeX */
	strarg = section->addStringOption("pdftex", GLE_TOOL_PDFTEX_CMD);
#ifdef __WIN32__
	strarg->setDefault("pdflatex.exe");
#else
	strarg->setDefault("pdflatex");
#endif
	/* DVIPS */
	strarg = section->addStringOption("dvips", GLE_TOOL_DVIPS_CMD);
#ifdef __WIN32__
	strarg->setDefault("dvips.exe");
#endif
#ifdef __UNIX__
	strarg->setDefault("dvips");
#endif
#ifdef __OS2__
	strarg->setDefault("dvips.exe");
#endif
	/* GhostScript */
	strarg = section->addStringOption("ghostscript", GLE_TOOL_GHOSTSCRIPT_CMD);
#ifdef __WIN32__
	strarg->setDefault("gswin32c.exe");
#endif
#ifdef __UNIX__
	strarg->setDefault("gs");
#endif
#ifdef __OS2__
	strarg->setDefault("gsos2.exe");
#endif
	collection->addSection(section, GLE_CONFIG_TOOLS);
/* TeX config */
	section = new ConfigSection("tex");
	option = new CmdLineOption("system");
	setarg = new CmdLineArgSet("device-names");
	setarg->setMaxCard(1);
	setarg->addPossibleValue("latex");
	setarg->addPossibleValue("vtex");
#ifdef __OS2__
	setarg->addDefaultValue(GLE_TEX_SYSTEM_VTEX);
#else
	setarg->addDefaultValue(GLE_TEX_SYSTEM_LATEX);
#endif
	option->addArg(setarg);
	section->addOption(option, GLE_TEX_SYSTEM);
	collection->addSection(section, GLE_CONFIG_TEX);
/* Config paper*/	
	section = new ConfigSection("paper");
	strarg = section->addStringOption("size", GLE_CONFIG_PAPER_SIZE);
	strarg->setDefault("a4paper");
	strarg = section->addStringOption("margins", GLE_CONFIG_PAPER_MARGINS);
	strarg->setDefault("2.54 2.54 2.54 2.54");
	collection->addSection(section, GLE_CONFIG_PAPER);
	collection->setDefaultValues();
}

const string& gle_config_margins() {
	ConfigSection* paper = g_Config.getSection(GLE_CONFIG_PAPER);
	return ((CmdLineArgString*)paper->getOptionValue(GLE_CONFIG_PAPER_MARGINS))->getValue();
}

const string& gle_config_papersize() {
	ConfigSection* paper = g_Config.getSection(GLE_CONFIG_PAPER);
	return ((CmdLineArgString*)paper->getOptionValue(GLE_CONFIG_PAPER_SIZE))->getValue();
}

#ifdef __WIN32__
bool gle_load_registry(const char* gle_key, string* gle_exe) {
	HKEY hk;
	if (RegOpenKeyEx(HKEY_CURRENT_USER, gle_key, 0, KEY_READ, &hk) == ERROR_SUCCESS) {
		unsigned char buffer[1024];
		DWORD dwtype, dwbsize = 1024;
		if (RegQueryValueEx(hk, "GLE_EXE", NULL, &dwtype, buffer, &dwbsize) == ERROR_SUCCESS) {
			*gle_exe = (char*)buffer;
		}
		RegCloseKey(hk);
		return true;
	} else {
		return false;
	}
}

void gle_save_registry(const char* gle_key, const string& gle_exe) {
	HKEY hk;
	DWORD dwDisp;
	if (RegCreateKeyEx(HKEY_CURRENT_USER, gle_key, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hk, &dwDisp)) {
		printf("Could not create a registry key for GLE.");
		return;
	}
	const char* val = gle_exe.c_str();
	RegSetValueEx(hk, "GLE_EXE", 0, REG_SZ, (const BYTE*)val, strlen(val)+1);
	RegCloseKey(hk);
}
#endif

bool try_load_config(const string& fname) {
	if (text_load(fname, GLESource)) {
		g_select_device(GLE_DEVICE_DUMMY);
		g_message_first_newline(false);
		DrawIt((char*)fname.c_str(), GLESource, NULL, true);
		GLESource.clear();
		return true;
	} else {
		return false;
	}
}

bool try_save_config(const string& fname, ConfigCollection* collection) {
	if (collection->allDefaults()) {
		cout << "Collection::All defaults" << endl;
		return true;
	}
	ofstream fout(fname.c_str());
	if (!fout.is_open()) {
		cout << ">>> Can't write to config file '" << fname << "'" << endl;
		return false;
	}
	for (int i = 0; i < collection->getNbSections(); i++) {
		ConfigSection* sec = collection->getSection(i);
		if (!sec->allDefaults()) {
			fout << "begin config " << sec->getName() << endl;
			for (int j = 0; j < sec->getNbOptions(); j++) {
				CmdLineOption* option = sec->getOption(j);
				if (!option->allDefaults()) {
					fout << "\t" << option->getName() << " = ";
					for (int k = 0; k < option->getMaxNbArgs(); k++) {
						if (k != 0) fout << " ";
						CmdLineOptionArg* arg = option->getArg(k);
						arg->write(fout);
					}
					fout << endl;
				}
			}
			fout << "end config" << endl << endl;
		}
	}
	fout.close();
	return true;
}

void find_deps(const string& loc, string* gleexe, ConfigCollection* collection) {
	vector<string> tofind;
	vector<string*> result;
#ifdef __WIN32__
	tofind.push_back("gle.exe");
	result.push_back(gleexe);
#endif
	ConfigSection* tools = collection->getSection(GLE_CONFIG_TOOLS);
	for (int i = 0; i < tools->getNbOptions(); i++) {
		CmdLineArgString* strarg = (CmdLineArgString*)tools->getOption(i)->getArg(0);
		tofind.push_back(strarg->getDefault());
		result.push_back(strarg->getValuePtr());
	}
	cout << "Finding dependencies in: " << loc << ": "; fflush(stdout);
	GLEFindFiles(loc, tofind, result);
	cout << endl;
	for (int i = 0; i < result.size(); i++) {
		CmdLineArgString* strarg = i > 0 ? (CmdLineArgString*)tools->getOption(i-1)->getArg(0) : NULL;
		if (result[i]->length() == 0 || (strarg != NULL && strarg->isDefault())) {
			cout << "Found: " << tofind[i] << " in '?'" << endl;
		} else {
			cout << "Found: " << tofind[i] << " in '" << *(result[i]) << "'" << endl;
		}
	}
}

void do_find_deps(CmdLineObj& cmdline) {
#ifdef __WIN32__
	string gle_exe;
	const char* gle_key = "Software\\GLE";
	gle_load_registry(gle_key, &gle_exe);
#endif
	// Set GLE_TOP
	// -> prefer environment var GLE_TOP
	// -> on windows, otherwise take value from registry (discovered by gle -finddeps)
	// -> on other systems, otherwise take install dir
	const char* top = getenv("GLE_TOP");
	if (top == NULL || top[0] == 0) {
		#ifdef __WIN32__
				if (gle_exe.length() != 0) {
					GLE_TOP_DIR = gle_exe;
					StripPathComponents(&GLE_TOP_DIR, 2);
				} else {
					GLE_TOP_DIR = INSTALL_DIR;
				}
		#else
			#ifdef INSTALL_DIR
				GLE_TOP_DIR = INSTALL_DIR;
			#else
				// The user will see as error messege: "$GLE_TOP/some_file" not found.
				GLE_TOP_DIR = "$GLE_TOP";
			#endif
		#endif
	} else {
		GLE_TOP_DIR = top;
	}
	// Try load config file
	// -> first $GLE_TOP/glerc
	// -> on Unix also $HOME/.glerc
	string conf_name = GLE_TOP_DIR + DIR_SEP + "glerc";
	bool has_config = try_load_config(conf_name);
	#if defined(__UNIX__) || defined (__OS2__)
		const char* home = getenv("HOME");
		if (home != NULL && home[0] != 0) {
			try_load_config(string(home) + DIR_SEP + ".glerc");
		}
	#endif
	// Save config if find deps
	if (cmdline.hasOption(GLE_OPT_FINDDEPS)) {
		CmdLineArgString* arg = (CmdLineArgString*)cmdline.getOption(GLE_OPT_FINDDEPS)->getArg(0);
		if (!cmdline.hasOption(GLE_OPT_INSTALL) || !has_config) {
#ifdef __WIN32__
			if (arg->getValue().length() > 0) find_deps(arg->getValue(), &gle_exe, &g_Config);
			else find_deps("C:\\Program Files", &gle_exe, &g_Config);
			gle_save_registry(gle_key, gle_exe);
			if (gle_exe.length() != 0) {
				GLE_TOP_DIR = gle_exe;
				StripPathComponents(&GLE_TOP_DIR, 2);
				conf_name = GLE_TOP_DIR + DIR_SEP + "glerc";
			}
			try_save_config(conf_name, &g_Config);
#endif
		}
		exit(0);
	}
}
