//
// the main section of gle
//
#include <math.h>
#include <stdlib.h>
#include <time.h>
#include <string>
#include <sstream>
#include <iostream>
#include <vector>
#include <list>
#include <map>
#include <fstream>

#ifdef __WIN32__
	#include <io.h>
#endif

using namespace std;

#include "SourceLine.h"
#include "all.h"
#include "tokens/Tokenizer.h"
#include "core.h"
#include "mem_limits.h"
#include "file_io.h"
#include "cutils.h"
#include "cmdline.h"
#include "texinterface.h"
#include "config.h"
#include "drawit.h"

/*------------ GLOBAL VARIABLES --------------*/
//
// -- these globals are needed. they are called in other source files using extern...
//
int MAX_VECTOR=MAXIMUM_PS_VECTOR;
int gle_debug;
bool control_d = true;
extern int trace_on;
bool BLACKANDWHITE = false;
bool IS_INSTALL = false;
bool GS_PREVIEW = false;

void do_find_deps(CmdLineObj& cmdline);
void gle_as_a_calculator(vector<string>* exprs);

string GLE_TOP_DIR;

_GLESource GLESource; // need global access

class GLEOptions {
public:
	bool ASK_DEBUG;
};

/* Better to put all global options in one class? */
GLEOptions g_Options;
CmdLineObj g_CmdLine;
ConfigCollection g_Config;

void load_one_file(const char* name, CmdLineObj& cmdline, size_t* exit_code);
void init_option_args(CmdLineObj& cmdline);
void process_option_args(CmdLineObj& cmdline, GLEOptions& options);

int main(int argc, char **argv) {
	/* saves some memory for emergencies - is this needed???? */
	init_memory();
	g_init();

	/* Init and process command line arguments */
	init_config(&g_Config);
	init_option_args(g_CmdLine);
	g_CmdLine.parse(argc, argv);

	/* Error in command line arguments? */
	if (g_CmdLine.hasError()) {
		exit(-1);
	}
	/* Has calculator option? */
	if (g_CmdLine.hasOption(GLE_OPT_CALC)) {
		if (g_CmdLine.getNbMainArgs() == 0) {
			gle_as_a_calculator(NULL);
		} else {
			gle_as_a_calculator(g_CmdLine.getMainArgs());
		}
		exit(0);
	}
	/* Find dependencies */
	do_find_deps(g_CmdLine);
	if (g_CmdLine.getNbMainArgs() == 0 || g_CmdLine.hasOption(GLE_OPT_HELP)) {
		cerr << "GLE version " << __VN__ << "." << __BN__ << endl;
		cerr << "Usage: gle [options] filename.gle" << endl;
		cerr << "More information: gle " << g_CmdLine.getOptionPrefix() << "help" << endl;
		if (g_CmdLine.hasOption(GLE_OPT_HELP)) g_CmdLine.showHelp(GLE_OPT_HELP);
		exit(0);
	}

	/* Process optional arguments */
	process_option_args(g_CmdLine, g_Options);

	/* iterate over "file" arguments */
	size_t exit_code = 0;
	for (int i = 0; i < g_CmdLine.getNbMainArgs(); i++) {
	  // the standard CMD shell didn't expand the wildcards
#ifdef __WIN32__
		//
		// -- globbing and wildcards
		//
		_finddata_t fileinfo;
		intptr_t File = _findfirst((char*)g_CmdLine.getMainArg(i).c_str(), &fileinfo);
		if (File != -1){
			do {
				load_one_file((char*)fileinfo.name, g_CmdLine, &exit_code);
			} while(!_findnext(File,&fileinfo));
			_findclose(File);
		} else {
			exit_code = 1;
			cout << "Can't open file: " << g_CmdLine.getMainArg(i) << endl;
		}
#else
#if defined(__OS2__) && defined(__EMX__)
	  char **list;
	  list = _fnexplode((char*)g_CmdLine.getMainArg(i).c_str());
	  if ( list == NULL ) {
	    load_one_file( g_CmdLine.getMainArg(i).c_str(), g_CmdLine, &exit_code );
	  } else {
	    for ( int i = 0; list[i] != NULL; i++ ) {
	      load_one_file(list[i], g_CmdLine, &exit_code);
	    }
	  }
	  _fnexplodefree(list);
#else
		// on Unix the shell expands the wildcards for us
		load_one_file(g_CmdLine.getMainArg(i).c_str(), g_CmdLine, &exit_code);
#endif
#endif
	}
	// will exit with the number of files it could NOT open
	exit(exit_code);
}

void process_option_args(CmdLineObj& cmdline, GLEOptions& options) {
	trace_on = cmdline.hasOption(GLE_OPT_TRACE);
	options.ASK_DEBUG = cmdline.hasOption(GLE_OPT_DEBUG);
	control_d = cmdline.hasOption(GLE_OPT_NO_CTRL_D);
	if (cmdline.hasOption(GLE_OPT_NO_MAXPATH)) {
		/* Huh - what does this do? */
		MAX_VECTOR = 10*MAXIMUM_PS_VECTOR;
	}
	BLACKANDWHITE = cmdline.hasOption(GLE_OPT_NO_COLOR);
	IS_INSTALL = cmdline.hasOption(GLE_OPT_INSTALL);
	if (cmdline.hasOption(GLE_OPT_BBTWEAK)) g_psbbtweak();
	GS_PREVIEW = cmdline.hasOption(GLE_OPT_GSPREVIEW);
}

void init_option_args(CmdLineObj& cmdline) {
	CmdLineOption* option;
	CmdLineArgString* strarg;
	CmdLineArgInt* intarg;
	CmdLineArgSet* setarg;
	cmdline.setMainArgType("file name");
	option = new CmdLineOption("help", "h", "?");
	option->setHelp("Shows help about command line options");
	strarg = new CmdLineArgString("option");
	strarg->setHelp("show specific help about 'option'");
	strarg->setCardLimits(0, 1);
	option->addArg(strarg);
	cmdline.addOption(option, GLE_OPT_HELP);
	option = new CmdLineOption("device", "d");
	option->setHelp("Select output device(s)");
	option->setMinNbArgs(1);
	setarg = new CmdLineArgSet("device-names");
	setarg->setHelp("set output device(s)");
	setarg->setMinCard(1);
	/* Order of values is important! (must match GLE_DEVICE_ constants in core.h) */
	setarg->addPossibleValue("eps");
	setarg->addPossibleValue("ps");
	setarg->addPossibleValue("pdf");
	setarg->addPossibleValue("svg");
	setarg->addPossibleValue("jpg");
	setarg->addPossibleValue("png");
	setarg->addPossibleValue("x11");
#ifndef GLEX11
	setarg->setUnsupportedValue(GLE_DEVICE_X11);
#endif
	setarg->addDefaultValue(GLE_DEVICE_EPS);
	option->addArg(setarg);
	cmdline.addOption(option, GLE_OPT_DEVICE);
	option = new CmdLineOption("inc");
	option->setHelp("Create .inc file for TeX code");
	cmdline.addOption(option, GLE_OPT_CREATE_INC);
	option = new CmdLineOption("fullpage");
	option->setHelp("Full page output");
	cmdline.addOption(option, GLE_OPT_FULL_PAGE);
	option = new CmdLineOption("nocolor", "bw");
	option->setHelp("Grayscale output");
	cmdline.addOption(option, GLE_OPT_NO_COLOR);
	option = new CmdLineOption("noctrl-d");
	option->setHelp("Do not include CTRL-D in PostScript output");
	cmdline.addOption(option, GLE_OPT_NO_CTRL_D);
	option = new CmdLineOption("dpi");
	option->setHelp("Set DPI value for bitmap import/export and .pdf output");
	intarg = new CmdLineArgInt("dpi");
	intarg->setHelp("set DPI value to 'dpi'");
	intarg->setCardLimits(0, 1);
	intarg->setDefault(72);
	option->addArg(intarg);
	cmdline.addOption(option, GLE_OPT_DPI);
	option = new CmdLineOption("texincprefix");
	option->setHelp("Adds given subdirectory to path in .inc file");
	strarg = new CmdLineArgString("path");
	strarg->setHelp("adds 'path' to path in .inc file");
	strarg->setCardLimits(1, 1);
	option->addArg(strarg);
	cmdline.addOption(option, GLE_OPT_TEXINCPREF);
	option = new CmdLineOption("nopdftex");
	option->setHelp("Disable PdfLaTeX for .pdf creation");
	cmdline.addOption(option, GLE_OPT_NO_PDFTEX);
#ifdef __WIN32__
	option = new CmdLineOption("finddeps");
	option->setHelp("Automatically finds dependencies");
	strarg = new CmdLineArgString("path");
	strarg->setHelp("find dependencies in 'path'");
	strarg->setCardLimits(0, 1);
	// Default value should also be set if no value is given?
	option->addArg(strarg);
	cmdline.addOption(option, GLE_OPT_FINDDEPS);
#endif
#ifdef ENABLE_GS_PREVIEW
	option = new CmdLineOption("gs");
	option->setHelp("Call ghostscript for previewing");
	cmdline.addOption(option, GLE_OPT_GSPREVIEW);
#endif
	option = new CmdLineOption("calc", "c");
	option->setHelp("Use GLE as a calculator");
	cmdline.addOption(option, GLE_OPT_CALC);
	option = new CmdLineOption("trace");
	option->setHelp("Trace GLE");
	option->setExpert(true);
	cmdline.addOption(option, GLE_OPT_TRACE);
	option = new CmdLineOption("debug");
	option->setHelp("Debug GLE");
	option->setExpert(true);
	cmdline.addOption(option, GLE_OPT_DEBUG);
	option = new CmdLineOption("nomaxpath");
	option->setHelp("No max path");
	option->setExpert(true);
	cmdline.addOption(option, GLE_OPT_NO_MAXPATH);
	option = new CmdLineOption("install");
	option->setHelp("Use during installation");
	option->setExpert(true);
	cmdline.addOption(option, GLE_OPT_INSTALL);
	cmdline.initOptions();
}

bool has_pdflatex(CmdLineObj& cmdline) {
	if (cmdline.hasOption(GLE_OPT_NO_PDFTEX)) {
		return false;
	}
	ConfigSection* tex = g_Config.getSection(GLE_CONFIG_TEX);
	CmdLineArgSet* texsys =	(CmdLineArgSet*)tex->getOptionValue(GLE_TEX_SYSTEM);
	if (texsys->hasValue(GLE_TEX_SYSTEM_VTEX)) {
		return false;
	} else {
		return true;
	}
}

bool has_eps_based_device(CmdLineArgSet* device) {
	if (device->hasValue(GLE_DEVICE_EPS)) return true;
	if (device->hasValue(GLE_DEVICE_PDF)) return true;
	if (device->hasValue(GLE_DEVICE_JPEG)) return true;
	if (device->hasValue(GLE_DEVICE_PNG)) return true;
	return false;
}

bool has_tex_eps_based_device_not_inc(CmdLineArgSet* device, CmdLineObj& cmdline) {
	if (!cmdline.hasOption(GLE_OPT_CREATE_INC)) {
		if (device->hasValue(GLE_DEVICE_EPS)) return true;
		if (device->hasValue(GLE_DEVICE_PDF) && !has_pdflatex(cmdline)) return true;
	}
	if (device->hasValue(GLE_DEVICE_JPEG)) return true;
	if (device->hasValue(GLE_DEVICE_PNG)) return true;
	return false;
}

bool requires_tex(CmdLineArgSet* device, CmdLineObj& cmdline) {
	if (!cmdline.hasOption(GLE_OPT_CREATE_INC)) {
		if (device->hasValue(GLE_DEVICE_EPS)) return true;
		if (device->hasValue(GLE_DEVICE_PDF)) return true;
	}
	if (device->hasValue(GLE_DEVICE_JPEG)) return true;
	if (device->hasValue(GLE_DEVICE_PNG)) return true;
	return false;
}

bool requires_temp_eps(CmdLineArgSet* device, CmdLineObj& cmdline) {
	if (!cmdline.hasOption(GLE_OPT_CREATE_INC)) {
		if (device->hasValue(GLE_DEVICE_PDF)) return true;
	}
	if (device->hasValue(GLE_DEVICE_JPEG)) return true;
	if (device->hasValue(GLE_DEVICE_PNG)) return true;
	return false;
}

bool has_bitmap_or_pdf_device(CmdLineArgSet* device) {
	if (device->hasValue(GLE_DEVICE_JPEG)) return true;
	if (device->hasValue(GLE_DEVICE_PNG)) return true;
	if (device->hasValue(GLE_DEVICE_PDF)) return true;
	return false;
}

bool is_bitmap_device(int device) {
	if (device == GLE_DEVICE_JPEG) return true;
	if (device == GLE_DEVICE_PNG) return true;
	return false;
}

bool process_one_file_eps(const char* name, CmdLineObj& cmdline) {
	CmdLineArgSet* device = (CmdLineArgSet*)cmdline.getOption(GLE_OPT_DEVICE)->getArg(0);
	g_select_device(GLE_DEVICE_EPS);
	/* Ghostscript likes the origin of the bounding box to be at (0,0) */
	if (has_bitmap_or_pdf_device(device)) {
		g_psbbtweak();
	}
	/* In some cases two passes are required to measure size of TeX objects */
	int done = 1;
	TeXInterface* interface = TeXInterface::getInstance();
	interface->initialize(name);
	do {
		interface->reset();
		DrawIt((char*)name, GLESource, &cmdline);
		done = interface->tryCreateHash();
	} while (done == 1);
	interface->checkObjectDimensions();
	if (cmdline.hasOption(GLE_OPT_CREATE_INC)) {
		CmdLineArgString* addinc = (CmdLineArgString*)cmdline.getOption(GLE_OPT_TEXINCPREF)->getArg(0);
		interface->createInc(addinc->getValue());
	}
	if (interface->hasObjects() && requires_tex(device, cmdline)) {
		interface->createTeX();
	}
	return interface->hasObjects();
}

void load_one_file(const char* name, CmdLineObj& cmdline, size_t* exit_code) {
	string str_name(name);
	if (text_load(name, GLESource)){
		g_message_first_newline(true);
		if (cmdline.hasOption(GLE_OPT_DEBUG)){
			printf("Debug options 16=do_pcode, 8=pass 4=polish, 2=eval ");
			printf("Debug "); scanf("%d",&gle_debug);
			printf("Trace "); scanf("%d",&trace_on);
		}
		bool create_inc = cmdline.hasOption(GLE_OPT_CREATE_INC);
		CmdLineArgSet* device = (CmdLineArgSet*)cmdline.getOption(GLE_OPT_DEVICE)->getArg(0);
		if (has_eps_based_device(device)) {
			bool has_tex = process_one_file_eps(name, cmdline);
			/* Return if there were errors */
			if (get_nb_errors() > 0) {
				return;
			}
			/* Get device information */
			PSGLEDevice* psdev = (PSGLEDevice*)g_get_device_ptr();
			int bb_wd = psdev->getBBWidth();
			int bb_hi = psdev->getBBHeight();
			int dpi = ((CmdLineArgInt*)cmdline.getOption(GLE_OPT_DPI)->getArg(0))->getValue();
			/* File has TeX code -> move it and create again */
			if (has_tex) {
				string main_name, from_name, to_name;
				GetMainName(str_name, main_name);
				from_name = main_name + ".eps";
				to_name = main_name + "_inc.eps";
				GLECopyFile(from_name, to_name);
				bool has_tex_file = false;
				if (has_tex_eps_based_device_not_inc(device, cmdline)) {
					has_tex_file = true;
					create_tex_eps_file(str_name);
				}
				if (device->hasValue(GLE_DEVICE_PDF)) {
					bool has_pdftex = has_pdflatex(cmdline);
					if (has_pdftex || create_inc) {
						create_pdf_file(to_name, dpi, bb_wd, bb_hi, false);
					}
					if (!create_inc) {
						has_tex_file = true;
						if (has_pdftex) {
							create_pdf_file_pdflatex(str_name);
						} else {
							create_pdf_file(str_name, dpi, bb_wd, bb_hi, true);
						}
					}
				}
				if (has_tex_file) DeleteFileWithExt(str_name, ".tex");
			} else {
				if (device->hasValue(GLE_DEVICE_PDF)) {
					create_pdf_file(str_name, dpi, bb_wd, bb_hi, false);
				}
			}
			/* Create bitmap outputs */
			bool is_bw = cmdline.hasOption(GLE_OPT_NO_COLOR);
			for (int i = 0; i < device->getNbValues(); i++) {
				if (is_bitmap_device(i) && device->hasValue(i)) {
					create_bitmap_file(str_name, i, dpi, bb_wd, bb_hi, is_bw, has_tex);
				}
			}
			/* Remove .eps if not required anymore */
			if (!device->hasValue(GLE_DEVICE_EPS)) {
				/* delete original .eps file */
				DeleteFileWithExt(str_name, ".eps");
			}
			if (has_tex) {
				if (create_inc) {
					/* delete original .eps file */
					DeleteFileWithExt(str_name, ".eps");
					if (!device->hasValue(GLE_DEVICE_EPS)) {
						DeleteFileWithExt(str_name, "_inc.eps");
					}
				} else {
					/* Remove _inc.{eps,pdf} if not required anymore */
					if (requires_tex(device, cmdline)) {
						DeleteFileWithExt(str_name, "_inc.eps");
					}
					if (device->hasValue(GLE_DEVICE_PDF)) {
						DeleteFileWithExt(str_name, "_inc.pdf");
					}
				}
				/* Remove _temp.eps? (used for bounding box adjustment) */
				if (requires_temp_eps(device, cmdline)) {
					DeleteFileWithExt(str_name, "_temp.eps");
				}
			}
		}
		if (device->hasValue(GLE_DEVICE_PS)) {
			g_select_device(GLE_DEVICE_PS);
			DrawIt((char*)name, GLESource, &cmdline);
		}
		if (device->hasValue(GLE_DEVICE_SVG)) {
			g_select_device(GLE_DEVICE_SVG);
			DrawIt((char*)name, GLESource, &cmdline);
		}
		if (device->hasValue(GLE_DEVICE_X11)) {
			g_select_device(GLE_DEVICE_X11);
			DrawIt((char*)name, GLESource, &cmdline);
		}
		GLESource.clear();
	} else {
		cout << "Can't open file: " << name << endl;
		(*exit_code)++;
	}
}
