
#include <string>
#include <vector>
#include <list>
using namespace std;

#include "tokens/stokenizer.h"
#include "SourceLine.h"
#include "all.h"
#include "cutils.h"
#include "gprint.h"
#include "numberformat.h"

/*
	dec, hex [upper|lower], bin
	01

	fix 3 + effect van nozeros
	12.000

	sig 2 [e,E,10,10+] expdigits 2 expsign

	round 3
	round number to 3 significant digits
	pad with zeros if number too large (don't use exponential represenations)

	nozeros

	sign
	always include sign in output (also for positive numbers)

	pad 7 [left|right]
	numbers less than 7 chars are padded from left with spaces

	prefix 3 (add zeros from left)
*/

class GLENumberFormatterFix : public GLENumberFormatter {
protected:
	int m_NbDecPlaces;
public:
	virtual ~GLENumberFormatterFix();
	virtual void parseOptions(GLENumberFormat* format);
	virtual void format(double number, string* output);
};

class GLENumberFormatterRound : public GLENumberFormatter {
protected:
	int m_Sig;
public:
	virtual ~GLENumberFormatterRound();
	virtual void parseOptions(GLENumberFormat* format);
	virtual void format(double number, string* output);
};

class GLENumberFormatterSci : public GLENumberFormatter {
protected:
	int m_Sig;
	int m_Mode;
	int m_ExpDigits;
	bool m_HasExpDigits;
	bool m_ExpSign;
public:
	GLENumberFormatterSci();
	virtual ~GLENumberFormatterSci();
	virtual void parseOptions(GLENumberFormat* format);
	virtual void format(double number, string* output);
	inline bool hasExpDigits() { return m_HasExpDigits; }
	inline int getExpDigits() { return m_ExpDigits; }
	void setExpDigits(int digits);
	inline bool hasExpSign() { return m_ExpSign; }
	inline void setExpSign(bool ExpSign) { m_ExpSign = ExpSign; }
};

class GLENumberFormatterInt : public GLENumberFormatter {
protected:
	int m_Mode;
	bool m_Upper;
public:
	GLENumberFormatterInt(int mode);
	virtual ~GLENumberFormatterInt();
	virtual void format(double number, string* output);
	virtual void parseOptions(GLENumberFormat* format);
	inline bool hasUpper() { return m_Upper; }
	inline void setUpper(bool Upper) { m_Upper = Upper; }
};

#define GLE_NF_INT_DEC 0
#define GLE_NF_INT_HEX 1
#define GLE_NF_INT_BIN 2

#define GLE_NF_SCI_SMALL_E 0
#define GLE_NF_SCI_BIG_E   1
#define GLE_NF_SCI_10      2

GLENumberFormatter::GLENumberFormatter() {
	m_Prefix = -1;
	m_NoZeroes = false;
	m_Sign = false;
	m_PadLeft = -1;
	m_PadRight = -1;
	m_HasMin = false;
	m_HasMax = false;
}

GLENumberFormatter::~GLENumberFormatter() {
}

void GLENumberFormatter::parseOptions(GLENumberFormat* format) {
}

void GLENumberFormatter::format(double number, string* output) {
}

bool GLENumberFormatter::appliesTo(double number) {
	if (hasMin() && number < getMin()) return false;
	if (hasMax() && number > getMax()) return false;
	return true;
}

void GLENumberFormatter::doAll(string* output) {
	doNoZeroes(output);
	doPrefix(output);
	doSign(output);
	doPadLeft(output);
	doPadRight(output);
}

void GLENumberFormatter::doPrefix(string* output) {
	if (hasPrefix()) {
		bool has_sign = false;
		int prefix = getPrefix();
		int length = output->length();
		string::size_type pos = output->rfind('.');
		if (pos == string::npos) pos = length;
		if (length > 0 && output->at(0) == '-') {
			prefix++;
			has_sign = true;
		}
		if (pos < prefix) {
			string zeros = has_sign ? "-" : "";
			for (int i = 0; i < prefix-pos; i++) {
				zeros += "0";
			}
			if (has_sign) {
				zeros += output->substr(1, length-1);
			} else {
				zeros += *output;
			}
			*output = zeros;
		}
	}
}

void GLENumberFormatter::doNoZeroes(string* output) {
	if (hasNoZeroes() && output->rfind('.') != string::npos) {
		int nbzero = 0;
		int length = output->length();
		int pos = length-1;
		while (pos >= 0 && output->at(pos) == '0') {
			pos--; nbzero++;
		}
		*output = output->substr(0, length-nbzero);
	}
}

void GLENumberFormatter::doSign(string* output) {
	if (hasSign()) {
		if (output->length() > 0 && output->at(0) != '-') {
			output->insert(0, "+");
		}
	}
}

void GLENumberFormatter::doPadLeft(string* output) {
	if (hasPadLeft()) {
		int count = getPadLeft() - output->length();
		str_prefix(count, ' ', output);
	}
}

void GLENumberFormatter::doPadRight(string* output) {
	if (hasPadRight()) {
		int count = getPadRight() - output->length();
		if (count > 0) {
			for (int i = 0; i < count; i++) {
				*output += " ";
			}
		}
	}
}

void GLENumberFormatter::setMin(double min) {
	m_Min = min;
	m_HasMin = true;
}

void GLENumberFormatter::setMax(double max) {
	m_Max = max;
	m_HasMax = true;
}

void GLENumberFormatter::setDefaults(GLENumberFormatter* def) {
	if (def->hasPrefix()) setPrefix(def->getPrefix());
	if (def->hasNoZeroes()) setNoZeroes(true);
	if (def->hasSign()) setSign(true);
	if (def->hasPadLeft()) setPadLeft(def->getPadLeft());
	if (def->hasPadRight()) setPadRight(def->getPadRight());
}

void GLENumberFormatter::formatSimple(double value, string* output, int prec, int* exp) {
	char format[20], result[100];
	double pos_num = fabs(value);
	if (pos_num == 0.0) {
		*exp = 0;
		*output = "0";
	} else {
		*exp = gle_double_digits(pos_num, prec);	
		if (prec > 0) {
			sprintf(format, "%%.%df", prec-1);
			sprintf(result, format, pos_num / pow(10.0, *exp));
		} else {
			result[0] = 0;
		}
		*output = result;
	}
}

GLENumberFormatterFix::~GLENumberFormatterFix() {
}

void GLENumberFormatterFix::parseOptions(GLENumberFormat* format) {
	m_NbDecPlaces = format->nextInt();
}

void GLENumberFormatterFix::format(double number, string* output) {
	char format[20], result[100];
	sprintf(format, "%%.%df", m_NbDecPlaces);
	sprintf(result, format, number);
	*output = result;
	doAll(output);
}

GLENumberFormatterSci::GLENumberFormatterSci() {
	m_Mode = GLE_NF_SCI_SMALL_E;
	m_HasExpDigits = false;
	m_ExpSign = false;
}

GLENumberFormatterSci::~GLENumberFormatterSci() {
}

void GLENumberFormatterSci::setExpDigits(int digits) {
	m_ExpDigits = digits;
	m_HasExpDigits = true;
}

void GLENumberFormatterSci::parseOptions(GLENumberFormat* format) {
	m_Sig = format->nextInt();
	while (format->hasMoreTokens()) {
		const string& tpe = format->nextToken();
		if (tpe == "e") {
			m_Mode = GLE_NF_SCI_SMALL_E;
			format->incTokens();
		} else if (tpe == "E") {
			m_Mode = GLE_NF_SCI_BIG_E;
			format->incTokens();
		} else if (tpe == "10") {
			m_Mode = GLE_NF_SCI_10;
			format->incTokens();
		} else if (tpe == "expdigits") {
			format->incTokens();
			setExpDigits(format->nextInt());
		} else if (tpe == "expsign") {
			format->incTokens();
			setExpSign(true);
		} else {
			break;
		}
	}
}

void GLENumberFormatterSci::format(double number, string* output) {
	int exp;
	string exp_str;
	formatSimple(number, output, m_Sig, &exp);
	int abs_exp = exp > 0 ? exp : -exp;
	gle_int_to_string(abs_exp, &exp_str);
	if (hasExpDigits()) {
		int nb_digits = gle_int_digits(abs_exp);
		str_prefix(getExpDigits()-nb_digits-1, '0', &exp_str);
	}
	if (exp < 0) exp_str.insert(0, "-");
	else if (hasExpSign()) exp_str.insert(0, "+");
	switch (m_Mode) {
		case GLE_NF_SCI_SMALL_E:
			*output += "e";
			*output += exp_str;
			break;
		case GLE_NF_SCI_BIG_E:
			*output += "E";
			*output += exp_str;
			break;
		case GLE_NF_SCI_10:
			if (output->length() == 0) *output = "10^{";
			else *output += "\\cdot 10^{";
			*output += exp_str;
			*output += "}";
			break;
	}
	if (number < 0.0) output->insert(0, "-");
	doAll(output);
}

GLENumberFormatterRound::~GLENumberFormatterRound() {
}

void GLENumberFormatterRound::parseOptions(GLENumberFormat* format) {
	m_Sig = format->nextInt();
}


void GLENumberFormatterRound::format(double number, string* output) {
	int exp;
	formatSimple(number, output, m_Sig, &exp);
	string::size_type pos = output->find('.');
	if (exp < 0) {
		if (pos != string::npos) output->erase(pos, 1);
		string prefix = "0.";
		for (int i = 0; i < -exp-1; i++) {
			prefix += "0";
		}
		*output = prefix + (*output);
	} else {
		if (pos != string::npos) {
			exp -= output->length() - pos - 1;
			output->erase(pos, 1);
			if (exp < 0) output->insert(output->length()+exp, ".");
		}
		for (int i = 0; i < exp; i++) {
			(*output) += "0";
		}
	}
	if (number < 0.0) output->insert(0, "-");
	doAll(output);
}

GLENumberFormatterInt::GLENumberFormatterInt(int mode) : GLENumberFormatter() {
	m_Mode = mode;
	m_Upper = true;
}

GLENumberFormatterInt::~GLENumberFormatterInt() {
}

void GLENumberFormatterInt::parseOptions(GLENumberFormat* format) {
	if (m_Mode == GLE_NF_INT_HEX) {
		const string& up = format->nextToken();
		if (up == "upper") {
			format->incTokens();
		} else if (up == "lower") {
			setUpper(false);
			format->incTokens();
		}
	}
}

void GLENumberFormatterInt::format(double number, string* output) {
	char result[100];
	int number_int = (int)floor(number + 0.5);
	switch (m_Mode) {
		case GLE_NF_INT_DEC:
			sprintf(result, "%d", number_int);
			*output = result;
			break;
		case GLE_NF_INT_HEX:
			if (hasUpper()) sprintf(result, "%X", number_int);
			else sprintf(result, "%x", number_int);
			*output = result;
			break;
		case GLE_NF_INT_BIN:
			gle_int_to_string_bin(number_int, output);
			break;
	}
	doAll(output);
}

GLENumberFormat::GLENumberFormat(const string& formats) :
		m_Separator(" \""), m_Tokens(formats, m_Separator) {
	GLENumberFormatter* previous = &m_Default;
	while (hasMoreTokens()) {
		const string& name = nextToken();
		GLENumberFormatter* format = NULL;
		if (name == "fix") {
			incTokens();
			format = new GLENumberFormatterFix();
		} else if (name == "dec") {
			incTokens();
			format = new GLENumberFormatterInt(GLE_NF_INT_DEC);
		} else if (name == "hex") {
			incTokens();
			format = new GLENumberFormatterInt(GLE_NF_INT_HEX);
		} else if (name == "bin") {
			incTokens();
			format = new GLENumberFormatterInt(GLE_NF_INT_BIN);
		} else if (name == "round") {
			incTokens();
			format = new GLENumberFormatterRound();
		} else if (name == "sci") {
			incTokens();
			format = new GLENumberFormatterSci();
		} else if (name == "prefix") {
			incTokens();
			previous->setPrefix(nextInt());
		} else if (name == "nozeroes") {
			incTokens();
			previous->setNoZeroes(true);
		} else if (name == "sign") {
			incTokens();
			previous->setSign(true);
		} else if (name == "pad") {
			incTokens();
			int padvalue = nextInt();
			const string& lr = nextToken();
			if (lr == "left") {
				previous->setPadLeft(padvalue);
				incTokens();
			} else if (lr == "right") {
				previous->setPadRight(padvalue);
				incTokens();
			}
		} else if (name == "min") {
			incTokens();
			previous->setMin(nextDouble());
		} else if (name == "max") {
			incTokens();
			previous->setMax(nextDouble());
		} else if (name == "otherwise") {
			incTokens();
		} else {
			gprint("Unknown specifier in number format string: '%s'", name.c_str());
			incTokens();
		}
		if (format != NULL) {
			format->setDefaults(&m_Default);
			format->parseOptions(this);
			addFormat(format);
			previous = format;
		}
	}
}

GLENumberFormat::~GLENumberFormat() {
	for (int i = 0; i < m_Format.size(); i++) {
		delete m_Format[i];
	}
}

void GLENumberFormat::format(double number, string* output) {
	for (int i = 0; i < m_Format.size(); i++) {
		if (m_Format[i]->appliesTo(number)) {
			m_Format[i]->format(number, output);
			return;
		}
	}
	*output = "ERR";
}

void GLENumberFormat::incTokens() {
	if (hasMoreTokens()) m_Tokens.next_token();
}

int GLENumberFormat::nextInt() {
	if (hasMoreTokens()) {
		const string& token = nextToken();
		int result = atoi(token.c_str());
		incTokens();
		return result;
	} else {
		return 0;
	}
}

double GLENumberFormat::nextDouble() {
	char* err = NULL;
	const string& token = nextToken();
	double result = strtod(token.c_str(), &err);
	incTokens();
	return result;
}

void format_number_to_string(char* out, const char* format, double value) {
	string result;
	GLENumberFormat fmt(format);
	fmt.format(value, &result);
	strcpy(out, result.c_str());
}
