/*
 * 2004 Jan Struyf
 *
 */

#ifndef INCLUDE_TEXINTERFACE
#define INCLUDE_TEXINTERFACE

class TeXHashObject {
protected:
	string m_Line;
	int m_HasDimensions, m_Used, m_NbLines;
	double m_Width, m_Height, m_Baseline;
public:
	TeXHashObject(const string& line);
	void outputMeasure(ostream& os);
	void outputLog(ostream& os);
	void outputLines(ostream& os);
	void addFirstLine(string* str);
	void setDimension(double width, double height, double baseline);
	inline const string& getLine() { return m_Line; }
	inline int hasDimensions() { return m_HasDimensions; }
	inline int isUsed() { return m_Used; }
	inline void use() { m_Used = 1; }
	inline double getWidth() { return m_Width; }
	inline double getHeight() { return m_Height; }
	inline double getBaseline() { return m_Baseline; }
	inline int getNbLines() { return m_NbLines; }
	inline void setNbLines(int nb) { m_NbLines = nb; }
};

#define TEX_OBJ_INF_HAS_COLOR    1
#define TEX_OBJ_INF_HAS_JUSTIFY  2
#define TEX_OBJ_INF_HAS_POSITION 4
#define TEX_OBJ_INF_DONT_PRINT   8

class TeXObjectInfo {
protected:
	int m_Status;
	colortyp m_Color;
	int m_Just;
	double m_Xp, m_Yp;
public:
	TeXObjectInfo();
	void setJustify(int just);
	void setPosition(double xp, double yp);
	void setColor(colortyp* color);
	void initializeAll();
	inline void setFlag(int flag) { m_Status |= flag; }
	inline int getFlags() { return m_Status; }
	inline int getJustify() { return m_Just; }
	inline double getXp() { return m_Xp; }
	inline double getYp() { return m_Yp; }
	inline colortyp* getColor() { return &m_Color; }
};

class TeXObject {
protected:
	double m_Xp, m_Yp, m_DXp, m_DYp, m_Angle;
	TeXHashObject* m_Object;
	colortyp m_Color;
public:
	TeXObject();
	void output(ostream& os);
	int isBlack();
	void getDimensions(double* x1, double *y1, double *x2, double *y2);
	inline void setAngle(double angle) { m_Angle = angle; }
	inline double getAngle() { return m_Angle; }
	inline void setXY(double x, double y) { m_Xp = x; m_Yp = y; }
	inline void setDeviceXY(double x, double y) { m_DXp = x; m_DYp = y; }
	inline void setObject(TeXHashObject* obj) { m_Object = obj; }
	inline 	TeXHashObject* getObject() { return m_Object; }
	inline double getWidth() { return m_Object->getWidth(); }
	inline double getHeight() { return m_Object->getHeight(); }
	inline double getDXp() { return m_DXp; }
	inline double getDYp() { return m_DYp; }
	inline const string& getLine() { return m_Object->getLine(); }
	inline int hasObject() { return m_Object != NULL; }
	inline void setColor(colortyp* color) { m_Color = *color; }
	inline colortyp* getColor() { return &m_Color; }
};

#define TEX_INTERFACE_HASH_LOADED_NONE    0
#define TEX_INTERFACE_HASH_LOADED_PARTIAL 1
#define TEX_INTERFACE_HASH_LOADED_FULL    2

class TeXInterface {
protected:
	vector<string> m_Preamble;
	static TeXInterface m_Instance;
	vector<TeXObject*> m_TeXObjects;
	vector<TeXHashObject*> m_TeXHash;
	string m_MainName, m_HashName, m_DocClass;
	int m_HashLoaded, m_HashModified;
public:
	TeXObject* draw(const char* str);
	TeXObject* draw(const char* str, int nblines);
	TeXObject* draw(const char* str, TeXObjectInfo& info, int nblines);
	TeXObject* drawObj(TeXHashObject* hobj, TeXObjectInfo& info);
	void checkObjectDimensions();
	int createObj(const char* str);
	void initialize(const char* fname);
	void reset();
	int tryCreateHash();
	void createInc(const string& prefix);
	void createTeX();
	int getHashObjectIndex(const string& line);
	TeXHashObject* getHashObject(const string& line);
	TeXHashObject* getHashObject(int idx);
	void resetPreamble();
	inline bool hasObjects() { return m_TeXObjects.size() != 0; }
	inline void addPreamble(const string& line) { m_Preamble.push_back(line); }
	inline int getNbPreamble() { return m_Preamble.size(); }
	inline const string& getPreamble(int i) { return m_Preamble[i]; }
	inline void setDocumentClass(const string& line) { m_DocClass = line; }
	inline void setDocumentClass(const char* line) { m_DocClass = line; }
	inline const string& getDocumentClass() { return m_DocClass; }
	static inline TeXInterface* getInstance() { return &m_Instance; }
protected:
	void writeInc(ostream& out, const char* prefix);
	void tryLoadHash();
	void loadTeXLines();
	void loadTeXPS();
	void createHiddenDir();
	void saveTeXLines();
	void saveTeXPS();
	void createTeXPS();
	void addHashObject(TeXHashObject* obj);
	void cleanUpObjects();
	void cleanUpHash();
	void createPreamble(ostream& tex_file);

};

bool create_tex_eps_file(const string& fname);
bool create_bitmap_file(const string& fname, int device, int dpi, int width, int height, bool bw, bool tex);
bool create_pdf_file(const string& fname, int dpi, int width, int height, bool tex);

bool run_ghostscript(const string& args);
bool run_latex(const string& dir, const string& file);
bool run_dvips(const string& file, const char* opts, bool eps);
bool create_pdf_file_pdflatex(const string& fname);

#endif
